<?php


/*
 * Copyright (c) 2019 Berkine Design
 * ============================================================================
 * 
 *
 * Main PHP File to send file to Amazon S3 bucket via Multi Part Upload Feature
 * 
 * 
 * ============================================================================
 * Original Filename: multi-part-file-upload.php
 * Last Modified Date: Mon, 25 Nov 2019
 *
 */

	# Include AWS S3 credentials and parameters
	require 'configs/start.php';
	require_once 'init.php';


	use Aws\S3\S3Client;  
	use Aws\S3\MultipartUploader;
	use Aws\Exception\MultipartUploadException;
	use Aws\Exception\AwsException;
	use Aws\Exception\CredentialsException;


	# Single File Upload Variables
	$public_url = '';					# Public file link
	$private_url = '';					# Private file link
	$private_url_duration = '';			# Private file link duration (valid only for set time)
	

	$response_array = [];
	
	
	# Maximum allowed file size to upload - can be changed as needed (make sure PHP ini file is updated accordingly)
	# Single File Upload size limit is set as 20 MB in Bytes
	$file_size_limit = 20971520;


	#============================================================================================
	#
	# Main Code for sending SINGLE FILE to Amazon S3 bucket and generating public/private links
	#
	#============================================================================================

	# Check if Single File is selected for File Upload
	if(isset($_FILES['multipart-file'])) {


		# Verify that file does not exceed maximum allowed size (also checked via JS to avoid extra load on the server)
		if ($_FILES['multipart-file']['size'] > $file_size_limit) {

			$single_error_message = "Please select a file less than 20MB in size.";			
		
		# If file within the allowed size, proceed with uplaoding to Amazon S3 Bucket
		} else {


			# File Specifications (Name, Size, TempName, Extension)
			$file_name = $_FILES['multipart-file']['name'];				  
			$file_size = $_FILES['multipart-file']['size'];
			$file_tmp_name = $_FILES['multipart-file']['tmp_name'];
			$file_extension = explode('.', $file_name);
			$file_extension = strtolower(end($file_extension));


			$result = $s3->createMultipartUpload([
			    'Bucket'       => $config['s3']['bucketName'],		# Name of the bucket to which the object is being uploaded
			    'Key'          => $file_name,
			    'StorageClass' => 'STANDARD',						# S3 Storage Type - Can be one of the follwoing: STANDARD|REDUCED_REDUNDANCY|GLACIER|STANDARD_IA|ONEZONE_IA|INTELLIGENT_TIERING|DEEP_ARCHIVE
			    'ACL'          => 'public-read',					# S3 Object Access Control List - Can be one of the following: private|public-read|public-read-write|authenticated-read|aws-exec-read|bucket-owner-read|bucket-owner-full-control
			    'ContentDisposition' => 'attachment'				# Allows you to download the file without opening it in the browser
			]);


			$uploadId = $result['UploadId'];


			# If "Public S3 File Link" radio button was selected:
			# ===================================================
			if ($_POST['radiobutton-multipart'] === 'public-link') {

				
				try {

				    $file = fopen($file_tmp_name, 'rb');
				    
				    $partNumber = 1;
				    
				    # Upload the file in parts.
				    while (!feof($file)) {

				        $result = $s3->uploadPart([
				            'Bucket'     => $config['s3']['bucketName'],			# (string, required) Name of the bucket to which the object is being uploaded
				            'Key'        => $file_name,								# (string, required) Key to use for the object being uploaded
				            'UploadId'   => $uploadId,
				            'PartNumber' => $partNumber,
				            'Body'       => fread($file, 5 * 1024 * 1024),			# (int, default=int(5242880)) Part size, in bytes, to use when doing a multipart upload. This must between 5 MB and 5 GB, inclusive.
				        ]);

				        $parts['Parts'][$partNumber] = [
				            'PartNumber' => $partNumber,
				            'ETag' => $result['ETag'],
				        ];

				        $partNumber++;

				    }

				    
				    fclose($file);


				    // Complete the multipart upload.
					$result = $s3->completeMultipartUpload([
					    'Bucket'   => $config['s3']['bucketName'],
					    'Key'      => $file_name,
					    'UploadId' => $uploadId,
					    'MultipartUpload' => $parts,
					]);

					

					# Status of the S3 upload (returns 200 if uploaded successfully)
					$status_code = $result['@metadata']['statusCode'];

				
					# If successfully uploaded to S3 Bucket
					if ($status_code === 200) {

						# Public URL of the Upload S3 object
						$public_url = $result['Location'];
						

						# Store as a session variable (needed for attaching to the email)
						$_SESSION['file-link'] = $public_url;
						

						$response_array['public-url'] = $public_url;
						$response_array['message'] = 'File Successfully Uploaded.';
						$response_array['status'] = 'success';
						
						header('Content-type: application/json');
						echo json_encode($response_array, JSON_UNESCAPED_SLASHES);		

					} else {

						$response_array['message'] = 'There was an Error during File Upload.';
						$response_array['status'] = 'error';
						
						header('Content-type: application/json');
						echo json_encode($response_array, JSON_UNESCAPED_SLASHES);

					}


				} catch (S3Exception $e) {

							$response_array['message'] = 'There was an error: ' . $e->getMessage();
							$response_array['status'] = 'error';
							
							header('Content-type: application/json');
							echo json_encode($response_array, JSON_UNESCAPED_SLASHES);						

				} catch (MultipartUploadException $e) {

						 	$response_array['message'] = 'There was an error: ' . $e->getMessage();
							$response_array['status'] = 'error';
							
							header('Content-type: application/json');
							echo json_encode($response_array, JSON_UNESCAPED_SLASHES);

				} 


									

			# If "Private S3 File Link" radio button was selected:
			# ===================================================
			} elseif ($_POST['radiobutton-multipart'] === 'private-link') {
				
				try {

				    $file = fopen($file_tmp_name, 'rb');
				    
				    $partNumber = 1;
				    
				    # Upload the file in parts.
				    while (!feof($file)) {

				        $result = $s3->uploadPart([
				            'Bucket'     => $config['s3']['bucketName'],			# (string, required) Name of the bucket to which the object is being uploaded
				            'Key'        => $file_name,									# (string, required) Key to use for the object being uploaded
				            'UploadId'   => $uploadId,
				            'PartNumber' => $partNumber,
				            'Body'       => fread($file, 5 * 1024 * 1024),			# (int, default=int(5242880)) Part size, in bytes, to use when doing a multipart upload. This must between 5 MB and 5 GB, inclusive.
				        ]);

				        $parts['Parts'][$partNumber] = [
				            'PartNumber' => $partNumber,
				            'ETag' => $result['ETag'],
				        ];

				        $partNumber++;

				    }

				    
				    fclose($file);


				    // Complete the multipart upload.
					$result = $s3->completeMultipartUpload([
					    'Bucket'   => $config['s3']['bucketName'],
					    'Key'      => $file_name,
					    'UploadId' => $uploadId,
					    'MultipartUpload' => $parts,
					]);

						
					# Status of the S3 upload (returns 200 if uploaded successfully)
					$status_code = $result['@metadata']['statusCode'];
					
					
					# If successfully uploaded to S3 Bucket
					if ($status_code === 200) {
						
						# Return uploaded S3 object
						$s3_object = $s3 -> getCommand('GetObject',[
							'Bucket' => $config['s3']['bucketName'],
							'Key' => $file_name
						]);
						

						# Create S3 Object Presigned Private URL with expiration time
						$request = $s3->createPresignedRequest($s3_object , strtotime($_POST['timer-multipart'] . ' minutes'));


						# Return URL of the S3 Object with Presigned Private URL
						$presigned_url = (string)$request->getUri();
						$private_url = $presigned_url;

						
						# Time notification until which private link is valid
						$time = date('Y-m-d H:i:s', strtotime($_POST['timer-multipart'] . ' minute'));
						

						# Time notification message for the user
						$private_url_duration = "*Private link valid for " . $_POST['timer-multipart'] . " minutes until " . $time . " GMT+1";							


						$response_array['private-url'] = $private_url;
						$response_array['private-url-duration'] = $private_url_duration;
						$response_array['message'] = 'File Successfully Uploaded.';	
						$response_array['status'] = 'success';
						

						header('Content-type: application/json');
						echo json_encode($response_array, JSON_UNESCAPED_SLASHES);

						# Store as session variables (needed for attaching to the email)
						$_SESSION['file-link'] = $private_url;
						$_SESSION['url-time'] = $private_url_duration;
						

					} else {

						$response_array['message'] = 'File was Not Uploaded Successfully.';
						$response_array['status'] = 'error';
						
						header('Content-type: application/json');
						echo json_encode($response_array, JSON_UNESCAPED_SLASHES);

					}
		
				} catch (S3Exception $e) {

							$response_array['message'] = 'There was an error: ' . $e->getMessage();
							$response_array['status'] = 'error';
							
							header('Content-type: application/json');
							echo json_encode($response_array, JSON_UNESCAPED_SLASHES);						

				} catch (MultipartUploadException $e) {

						 	$response_array['message'] = 'There was an error: ' . $e->getMessage();
							$response_array['status'] = 'error';
							
							header('Content-type: application/json');
							echo json_encode($response_array, JSON_UNESCAPED_SLASHES);

				}


			} 


		
		}

	}


	
	
?>	